# - Find python libraries
# This module finds if Python is installed and determines where the
# include files and libraries are. It also determines what the name of
# the library is. This code sets the following variables:
#
#  PYTHON2LIBS_FOUND           - have the Python libs been found
#  PYTHON2_LIBRARIES           - path to the python library
#  PYTHON2_INCLUDE_PATH        - path to where Python.h is found (deprecated)
#  PYTHON2_INCLUDE_DIRS        - path to where Python.h is found
#  PYTHON2_DEBUG_LIBRARIES     - path to the debug library (deprecated)
#  PYTHON2LIBS_VERSION_STRING  - version of the Python libs found (since CMake 2.8.8)
#
# The Python_ADDITIONAL_VERSIONS variable can be used to specify a list of
# version numbers that should be taken into account when searching for Python.
# You need to set this variable before calling find_package(Python2Libs).
#
# If you'd like to specify the installation of Python to use, you should modify
# the following cache variables:
#  PYTHON2_LIBRARY             - path to the python library
#  PYTHON2_INCLUDE_DIR         - path to where Python.h is found

#=============================================================================
# Copyright 2001-2009 Kitware, Inc.
#
# Distributed under the OSI-approved BSD License (the "License");
# see accompanying file Copyright.txt for details.
#
# This software is distributed WITHOUT ANY WARRANTY; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the License for more information.
#=============================================================================
# (To distribute this file outside of CMake, substitute the full
#  License text for the above reference.)

INCLUDE(CMakeFindFrameworks)
# Search for the python framework on Apple.
CMAKE_FIND_FRAMEWORKS(Python)

SET(_PYTHON21_VERSIONS 1.6 1.5)
SET(_PYTHON22_VERSIONS 2.7 2.6 2.5 2.4 2.3 2.2 2.1 2.0)
SET(_PYTHON23_VERSIONS 3.3 3.2 3.1 3.0)

IF(Python2Libs_FIND_VERSION)
    IF(Python2Libs_FIND_VERSION MATCHES "^[0-9]+\\.[0-9]+(\\.[0-9]+.*)?$")
        STRING(REGEX REPLACE "^([0-9]+\\.[0-9]+).*" "\\1" _PYTHON2_FIND_MAJ_MIN "${Python2Libs_FIND_VERSION}")
        STRING(REGEX REPLACE "^([0-9]+).*" "\\1" _PYTHON2_FIND_MAJ "${_PYTHON2_FIND_MAJ_MIN}")
        UNSET(_PYTHON2_FIND_OTHER_VERSIONS)
        IF(Python2Libs_FIND_VERSION_EXACT)
            IF(_PYTHON2_FIND_MAJ_MIN STREQUAL Python2Libs_FIND_VERSION)
                SET(_PYTHON2_FIND_OTHER_VERSIONS "${Python2Libs_FIND_VERSION}")
            ELSE(_PYTHON2_FIND_MAJ_MIN STREQUAL Python2Libs_FIND_VERSION)
                SET(_PYTHON2_FIND_OTHER_VERSIONS "${Python2Libs_FIND_VERSION}" "${_PYTHON2_FIND_MAJ_MIN}")
            ENDIF(_PYTHON2_FIND_MAJ_MIN STREQUAL Python2Libs_FIND_VERSION)
        ELSE(Python2Libs_FIND_VERSION_EXACT)
            FOREACH(_PYTHON2_V ${_PYTHON2${_PYTHON2_FIND_MAJ}_VERSIONS})
                IF(NOT _PYTHON2_V VERSION_LESS _PYTHON2_FIND_MAJ_MIN)
                    LIST(APPEND _PYTHON2_FIND_OTHER_VERSIONS ${_PYTHON2_V})
                ENDIF()
             ENDFOREACH()
        ENDIF(Python2Libs_FIND_VERSION_EXACT)
        UNSET(_PYTHON2_FIND_MAJ_MIN)
        UNSET(_PYTHON2_FIND_MAJ)
    ELSE(Python2Libs_FIND_VERSION MATCHES "^[0-9]+\\.[0-9]+(\\.[0-9]+.*)?$")
        SET(_PYTHON2_FIND_OTHER_VERSIONS ${_PYTHON2${Python2Libs_FIND_VERSION}_VERSIONS})
    ENDIF(Python2Libs_FIND_VERSION MATCHES "^[0-9]+\\.[0-9]+(\\.[0-9]+.*)?$")
ELSE(Python2Libs_FIND_VERSION)
    SET(_PYTHON2_FIND_OTHER_VERSIONS ${_PYTHON23_VERSIONS} ${_PYTHON22_VERSIONS} ${_PYTHON21_VERSIONS})
ENDIF(Python2Libs_FIND_VERSION)

# Set up the versions we know about, in the order we will search. Always add
# the user supplied additional versions to the front.
SET(_Python_VERSIONS
  ${Python_ADDITIONAL_VERSIONS}
  ${_PYTHON2_FIND_OTHER_VERSIONS}
  )

UNSET(_PYTHON2_FIND_OTHER_VERSIONS)
UNSET(_PYTHON21_VERSIONS)
UNSET(_PYTHON22_VERSIONS)
UNSET(_PYTHON23_VERSIONS)

FOREACH(_CURRENT_VERSION ${_Python_VERSIONS})
  STRING(REPLACE "." "" _CURRENT_VERSION_NO_DOTS ${_CURRENT_VERSION})
  IF(WIN32)
    FIND_LIBRARY(PYTHON2_DEBUG_LIBRARY
      NAMES python${_CURRENT_VERSION_NO_DOTS}_d python
      PATHS
      [HKEY_LOCAL_MACHINE\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs/Debug
      [HKEY_CURRENT_USER\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs/Debug
      [HKEY_LOCAL_MACHINE\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs
      [HKEY_CURRENT_USER\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs
      )
  ENDIF(WIN32)

  FIND_LIBRARY(PYTHON2_LIBRARY
    NAMES
    python${_CURRENT_VERSION_NO_DOTS}
    python${_CURRENT_VERSION}mu
    python${_CURRENT_VERSION}m
    python${_CURRENT_VERSION}u
    python${_CURRENT_VERSION}
    PATHS
      [HKEY_LOCAL_MACHINE\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs
      [HKEY_CURRENT_USER\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/libs
    # Avoid finding the .dll in the PATH.  We want the .lib.
    NO_SYSTEM_ENVIRONMENT_PATH
  )
  # Look for the static library in the Python config directory
  FIND_LIBRARY(PYTHON2_LIBRARY
    NAMES python${_CURRENT_VERSION_NO_DOTS} python${_CURRENT_VERSION}
    # Avoid finding the .dll in the PATH.  We want the .lib.
    NO_SYSTEM_ENVIRONMENT_PATH
    # This is where the static library is usually located
    PATH_SUFFIXES python${_CURRENT_VERSION}/config
  )

  # For backward compatibility, honour value of PYTHON2_INCLUDE_PATH, if
  # PYTHON2_INCLUDE_DIR is not set.
  IF(DEFINED PYTHON2_INCLUDE_PATH AND NOT DEFINED PYTHON2_INCLUDE_DIR)
    SET(PYTHON2_INCLUDE_DIR "${PYTHON2_INCLUDE_PATH}" CACHE PATH
      "Path to where Python.h is found" FORCE)
  ENDIF(DEFINED PYTHON2_INCLUDE_PATH AND NOT DEFINED PYTHON2_INCLUDE_DIR)

  SET(PYTHON2_FRAMEWORK_INCLUDES)
  IF(Python_FRAMEWORKS AND NOT PYTHON2_INCLUDE_DIR)
    FOREACH(dir ${Python_FRAMEWORKS})
      SET(PYTHON2_FRAMEWORK_INCLUDES ${PYTHON2_FRAMEWORK_INCLUDES}
        ${dir}/Versions/${_CURRENT_VERSION}/include/python${_CURRENT_VERSION})
    ENDFOREACH(dir)
  ENDIF(Python_FRAMEWORKS AND NOT PYTHON2_INCLUDE_DIR)

  FIND_PATH(PYTHON2_INCLUDE_DIR
    NAMES Python.h
    PATHS
      ${PYTHON2_FRAMEWORK_INCLUDES}
      [HKEY_LOCAL_MACHINE\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/include
      [HKEY_CURRENT_USER\\SOFTWARE\\Python\\PythonCore\\${_CURRENT_VERSION}\\InstallPath]/include
    PATH_SUFFIXES
      python${_CURRENT_VERSION}mu
      python${_CURRENT_VERSION}m
      python${_CURRENT_VERSION}u
      python${_CURRENT_VERSION}
  )

  # For backward compatibility, set PYTHON2_INCLUDE_PATH.
  SET(PYTHON2_INCLUDE_PATH "${PYTHON2_INCLUDE_DIR}")

  IF(PYTHON2_INCLUDE_DIR AND EXISTS "${PYTHON2_INCLUDE_DIR}/patchlevel.h")
    FILE(STRINGS "${PYTHON2_INCLUDE_DIR}/patchlevel.h" python_version_str
         REGEX "^#define[ \t]+PY_VERSION[ \t]+\"[^\"]+\"")
    STRING(REGEX REPLACE "^#define[ \t]+PY_VERSION[ \t]+\"([^\"]+)\".*" "\\1"
                         PYTHON2LIBS_VERSION_STRING "${python_version_str}")
    UNSET(python_version_str)
  ENDIF(PYTHON2_INCLUDE_DIR AND EXISTS "${PYTHON2_INCLUDE_DIR}/patchlevel.h")

  IF(PYTHON2_LIBRARY AND PYTHON2_INCLUDE_DIR)
    BREAK()
  ENDIF(PYTHON2_LIBRARY AND PYTHON2_INCLUDE_DIR)
ENDFOREACH(_CURRENT_VERSION)

MARK_AS_ADVANCED(
  PYTHON2_DEBUG_LIBRARY
  PYTHON2_LIBRARY
  PYTHON2_INCLUDE_DIR
)

# We use PYTHON2_INCLUDE_DIR, PYTHON2_LIBRARY and PYTHON2_DEBUG_LIBRARY for the
# cache entries because they are meant to specify the location of a single
# library. We now set the variables listed by the documentation for this
# module.
SET(PYTHON2_INCLUDE_DIRS "${PYTHON2_INCLUDE_DIR}")
SET(PYTHON2_DEBUG_LIBRARIES "${PYTHON2_DEBUG_LIBRARY}")

# These variables have been historically named in this module different from
# what SELECT_LIBRARY_CONFIGURATIONS() expects.
SET(PYTHON2_LIBRARY_DEBUG "${PYTHON2_DEBUG_LIBRARY}")
SET(PYTHON2_LIBRARY_RELEASE "${PYTHON2_LIBRARY}")
INCLUDE(SelectLibraryConfigurations)
SELECT_LIBRARY_CONFIGURATIONS(PYTHON2)
# SELECT_LIBRARY_CONFIGURATIONS() sets ${PREFIX}_FOUND if it has a library.
# Unset this, this prefix doesn't match the module prefix, they are different
# for historical reasons.
UNSET(PYTHON2_FOUND)

INCLUDE(FindPackageHandleStandardArgs)
FIND_PACKAGE_HANDLE_STANDARD_ARGS(Python2Libs
                                  REQUIRED_VARS PYTHON2_LIBRARIES PYTHON2_INCLUDE_DIRS
                                  VERSION_VAR PYTHON2LIBS_VERSION_STRING)

# PYTHON2_ADD_MODULE(<name> src1 src2 ... srcN) is used to build modules for python.
# PYTHON2_WRITE_MODULES_HEADER(<filename>) writes a header file you can include
# in your sources to initialize the static python modules
FUNCTION(PYTHON2_ADD_MODULE _NAME )
  GET_PROPERTY(_TARGET_SUPPORTS_SHARED_LIBS
    GLOBAL PROPERTY TARGET_SUPPORTS_SHARED_LIBS)
  OPTION(PYTHON2_ENABLE_MODULE_${_NAME} "Add module ${_NAME}" TRUE)
  OPTION(PYTHON2_MODULE_${_NAME}_BUILD_SHARED
    "Add module ${_NAME} shared" ${_TARGET_SUPPORTS_SHARED_LIBS})

  # Mark these options as advanced
  MARK_AS_ADVANCED(PYTHON2_ENABLE_MODULE_${_NAME}
    PYTHON2_MODULE_${_NAME}_BUILD_SHARED)

  IF(PYTHON2_ENABLE_MODULE_${_NAME})
    IF(PYTHON2_MODULE_${_NAME}_BUILD_SHARED)
      SET(PY_MODULE_TYPE MODULE)
    ELSE(PYTHON2_MODULE_${_NAME}_BUILD_SHARED)
      SET(PY_MODULE_TYPE STATIC)
      SET_PROPERTY(GLOBAL  APPEND  PROPERTY  PY_STATIC_MODULES_LIST ${_NAME})
    ENDIF(PYTHON2_MODULE_${_NAME}_BUILD_SHARED)

    SET_PROPERTY(GLOBAL  APPEND  PROPERTY  PY_MODULES_LIST ${_NAME})
    ADD_LIBRARY(${_NAME} ${PY_MODULE_TYPE} ${ARGN})
#    TARGET_LINK_LIBRARIES(${_NAME} ${PYTHON2_LIBRARIES})

    IF(PYTHON2_MODULE_${_NAME}_BUILD_SHARED)
      SET_TARGET_PROPERTIES(${_NAME} PROPERTIES PREFIX "${PYTHON2_MODULE_PREFIX}")
      IF(WIN32 AND NOT CYGWIN)
        SET_TARGET_PROPERTIES(${_NAME} PROPERTIES SUFFIX ".pyd")
      ENDIF(WIN32 AND NOT CYGWIN)
    ENDIF(PYTHON2_MODULE_${_NAME}_BUILD_SHARED)

  ENDIF(PYTHON2_ENABLE_MODULE_${_NAME})
ENDFUNCTION(PYTHON2_ADD_MODULE)

FUNCTION(PYTHON2_WRITE_MODULES_HEADER _filename)

  GET_PROPERTY(PY_STATIC_MODULES_LIST  GLOBAL  PROPERTY PY_STATIC_MODULES_LIST)

  GET_FILENAME_COMPONENT(_name "${_filename}" NAME)
  STRING(REPLACE "." "_" _name "${_name}")
  STRING(TOUPPER ${_name} _nameUpper)
  SET(_filename ${CMAKE_CURRENT_BINARY_DIR}/${_filename})

  SET(_filenameTmp "${_filename}.in")
  FILE(WRITE ${_filenameTmp} "/*Created by cmake, do not edit, changes will be lost*/\n")
  FILE(APPEND ${_filenameTmp}
"#ifndef ${_nameUpper}
#define ${_nameUpper}

#include <Python.h>

#ifdef __cplusplus
extern \"C\" {
#endif /* __cplusplus */

")

  FOREACH(_currentModule ${PY_STATIC_MODULES_LIST})
    FILE(APPEND ${_filenameTmp} "extern void init${PYTHON2_MODULE_PREFIX}${_currentModule}(void);\n\n")
  ENDFOREACH(_currentModule ${PY_STATIC_MODULES_LIST})

  FILE(APPEND ${_filenameTmp}
"#ifdef __cplusplus
}
#endif /* __cplusplus */

")


  FOREACH(_currentModule ${PY_STATIC_MODULES_LIST})
    FILE(APPEND ${_filenameTmp} "int ${_name}_${_currentModule}(void) \n{\n  static char name[]=\"${PYTHON2_MODULE_PREFIX}${_currentModule}\"; return PyImport_AppendInittab(name, init${PYTHON2_MODULE_PREFIX}${_currentModule});\n}\n\n")
  ENDFOREACH(_currentModule ${PY_STATIC_MODULES_LIST})

  FILE(APPEND ${_filenameTmp} "void ${_name}_LoadAllPythonModules(void)\n{\n")
  FOREACH(_currentModule ${PY_STATIC_MODULES_LIST})
    FILE(APPEND ${_filenameTmp} "  ${_name}_${_currentModule}();\n")
  ENDFOREACH(_currentModule ${PY_STATIC_MODULES_LIST})
  FILE(APPEND ${_filenameTmp} "}\n\n")
  FILE(APPEND ${_filenameTmp} "#ifndef EXCLUDE_LOAD_ALL_FUNCTION\nvoid CMakeLoadAllPythonModules(void)\n{\n  ${_name}_LoadAllPythonModules();\n}\n#endif\n\n#endif\n")

# with CONFIGURE_FILE() cmake complains that you may not use a file created using FILE(WRITE) as input file for CONFIGURE_FILE()
  EXECUTE_PROCESS(COMMAND ${CMAKE_COMMAND} -E copy_if_different "${_filenameTmp}" "${_filename}" OUTPUT_QUIET ERROR_QUIET)

ENDFUNCTION(PYTHON2_WRITE_MODULES_HEADER)
