/***************************************************************************
            fstab.c  -  Access the /etc/fstab file
                             -------------------
    begin                : Mon Dec 26 2004
    copyright            : (C) 2004 by Markus Raab
    email                : elektra@markus-raab.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the BSD License (revised).                      *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This is a backend that takes /etc/fstab file as its backend storage.  *
 *   The kdbGet() method will parse /etc/fstab and generate a              *
 *   valid key tree. The kdbSet() method will take a KeySet with valid     *
 *   filesystem keys and print an equivalent regular fstab in stdout.      *
 *                                                                         *
 ***************************************************************************/

#include "fstab.h"

#define MAX_NUMBER_SIZE 10

/** @param name is a buffer with KDB_MAX_PATH_LENGTH space.
  * @param fstabEntry will be used to get the name:
  * @param swapIndex will count up for every swap
  *
  *   - mnt_type will be checked if it is swap
  *
  * TODO Improvements:
  *   - no counting up of swap?
  *   - handle mountpoints none?
  *
  * Some logic to define the filesystem name when it is not
  * so obvious.
  */
void elektraFstabFsName(char * fsname, struct mntent *fstabEntry,
		unsigned int *swapIndex)
{

	if (!strcmp(fstabEntry->mnt_type,"swap")) {
		sprintf(fsname,"swap%02d",*swapIndex);
		++(*swapIndex);
	} else if (!strcmp(fstabEntry->mnt_dir,"none")) {
		strcpy(fsname,fstabEntry->mnt_type);
	} else if (!strcmp(fstabEntry->mnt_dir,"/")) {
		strcpy(fsname,"rootfs");
	} else {
		/* fsname will be the mount point without '/' char */
		char *slash=0;
		char *curr=fstabEntry->mnt_dir;
		fsname[0]=0;
		
		while((slash=strchr(curr,KDB_PATH_SEPARATOR))) {
			if (slash==curr) {
				curr++;
				continue;
			}
			
			strncat(fsname,curr,slash-curr);
			curr=slash+1;
		}
		strcat(fsname,curr);
	}
}

int elektraFstabGet(Plugin *handle, KeySet *returned, Key *parentKey)
{
	int errnosave = errno;
	ssize_t nr_keys = 0;
	Key *key;
	Key *dir;
	FILE *fstab=0;

#if DEBUG && VERBOSE
	printf ("get fstab %s from %s\n", keyName(parentKey), keyString(parentKey));
#endif

	if (!strcmp (keyName(parentKey), "system/elektra/modules/fstab"))
	{
		KeySet *moduleConfig = ksNew (50,
			keyNew ("system/elektra/modules/fstab",
				KEY_VALUE, "fstab plugin waits for your orders", KEY_END),
			keyNew ("system/elektra/modules/fstab/exports", KEY_END),
			keyNew ("system/elektra/modules/fstab/exports/get",
				KEY_FUNC, elektraFstabGet,
				KEY_END),
			keyNew ("system/elektra/modules/fstab/exports/set",
				KEY_FUNC, elektraFstabSet,
				KEY_END),
			keyNew ("system/elektra/modules/fstab/infos",
				KEY_VALUE, "All information you want to know", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/author",
				KEY_VALUE, "Markus Raab <elektra@markus-raab.org>", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/licence",
				KEY_VALUE, "BSD", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/description",
				KEY_VALUE, "/Parses files in a syntax like /etc/fstab file", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/provides",
				KEY_VALUE, "storage", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/placements",
				KEY_VALUE, "getstorage setstorage", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/needs",
				KEY_VALUE, "", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/recommends",
				KEY_VALUE, "struct type path", KEY_END),
			keyNew ("system/elektra/modules/fstab/infos/version",
				KEY_VALUE, PLUGINVERSION, KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs",
				KEY_VALUE, "The configuration which is needed",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct",
				KEY_VALUE, "list FStab",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab",
				KEY_META, "check/type", "null empty",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/device",
				KEY_META, "check/type", "string",
				KEY_META, "check/path", "device",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/mpoint",
				KEY_META, "check/type", "string",
				KEY_META, "check/path", "directory",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/type",
				KEY_META, "check/type", "FSType",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/options",
				KEY_META, "check/type", "string",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/dumpfreq",
				KEY_META, "check/type", "unsigned_short",
				KEY_END),
			keyNew ("system/elektra/modules/fstab/config/needs/struct/FStab/passno",
				KEY_META, "check/type", "unsigned_short",
				KEY_END),
			KS_END);
		ksAppend (returned, moduleConfig);
		ksDel (moduleConfig);
		return 1;
	}

	key = keyDup (parentKey);
	ksAppendKey(returned, key);
	nr_keys ++;

	fstab=setmntent(keyString(parentKey), "r");
	if (fstab == 0)
	{
		/* propagate errno */
		errno = errnosave;
		return -1;
	}

	struct mntent *fstabEntry;
	char fsname[KDB_MAX_PATH_LENGTH];
	char buffer[MAX_NUMBER_SIZE];
	unsigned int swapIndex=0;
	while ((fstabEntry=getmntent(fstab)))
	{
		nr_keys += 7;
		elektraFstabFsName(fsname, fstabEntry, &swapIndex);

		/* Include only the filesystem pseudo-names */
		dir = keyDup (parentKey);
		keyAddBaseName(dir, fsname);
		keySetString(dir,"");
		keySetComment(dir,"");
		keySetComment (dir, "Filesystem pseudo-name");
		ksAppendKey(returned,dir);

		key = keyDup (dir);
		keyAddBaseName(key, "device");
		keySetString (key, fstabEntry->mnt_fsname);
		keySetComment (key, "Device or Label");
		ksAppendKey(returned, key);

		key = keyDup (dir);
		keyAddBaseName(key, "mpoint");
		keySetString (key, fstabEntry->mnt_dir);
		keySetComment (key, "Mount point");
		ksAppendKey(returned, key);

		key = keyDup (dir);
		keyAddBaseName(key, "type");
		keySetString (key, fstabEntry->mnt_type);
		keySetComment (key, "Filesystem type.");
		ksAppendKey(returned, key);

		key = keyDup (dir);
		keyAddBaseName(key, "options");
		keySetString (key, fstabEntry->mnt_opts);
		keySetComment (key, "Filesystem specific options");
		ksAppendKey(returned, key);

		key = keyDup (dir);
		keyAddBaseName(key, "dumpfreq");
		snprintf(buffer, MAX_NUMBER_SIZE, "%d",fstabEntry->mnt_freq);
		keySetString (key, buffer);
		keySetComment (key, "Dump frequency in days");
		ksAppendKey(returned, key);

		key = keyDup (dir);
		keyAddBaseName(key, "passno");
		snprintf(buffer, MAX_NUMBER_SIZE, "%d",fstabEntry->mnt_passno);
		keySetString (key, buffer);
		keySetComment (key, "Pass number on parallel fsck");
		ksAppendKey(returned, key);
	}
	
	endmntent(fstab);

	errno = errnosave;
	return nr_keys;
}


int elektraFstabSet(Plugin *handle, KeySet *ks, Key *parentKey)
{
	int ret = 1;
	int errnosave = errno;
	FILE *fstab=0;
	Key *key=0;
	char *basename = 0;
	const void *rootname = 0;
	struct mntent fstabEntry;

#if DEBUG && VERBOSE
	printf ("set fstab %s to file %s\n", keyName(parentKey), keyString(parentKey));
#endif

	ksRewind (ks);
	if ((key = ksNext (ks)) != 0)
	{
		/*skip parent key*/
	}

	fstab=setmntent(keyString(parentKey), "w");
	memset(&fstabEntry,0,sizeof(struct mntent));

	while ((key = ksNext (ks)) != 0)
	{
		ret ++;
		basename=strrchr(keyName(key), '/')+1;
#if DEBUG && VERBOSE
		printf ("key: %s %s\n", keyName(key), basename);
#endif
		if (!strcmp (basename, "device"))
		{
			fstabEntry.mnt_fsname=(char *)keyValue(key);
		} else if (!strcmp (basename, "mpoint")) {
			fstabEntry.mnt_dir=(char *)keyValue(key);
		} else if (!strcmp (basename, "type")) {
			fstabEntry.mnt_type=(char *)keyValue(key);
		} else if (!strcmp (basename, "options")) {
			fstabEntry.mnt_opts=(char *)keyValue(key);
		} else if (!strcmp (basename, "dumpfreq")) {
			fstabEntry.mnt_freq=atoi((char *)keyValue(key));
		} else if (!strcmp (basename, "passno")) {
			fstabEntry.mnt_passno=atoi((char *)keyValue(key));
		} else { // new rootname
			if (!rootname)
			{
				rootname = keyValue(key);
			} else {
				rootname = keyValue(key);
#if DEBUG && VERBOSE
				fprintf(stdout, "first: %s   %s   %s   %s   %d %d\n",
					fstabEntry.mnt_fsname,
					fstabEntry.mnt_dir,
					fstabEntry.mnt_type,
					fstabEntry.mnt_opts,
					fstabEntry.mnt_freq,
					fstabEntry.mnt_passno);
#endif
				addmntent(fstab, &fstabEntry);
				memset(&fstabEntry,0,sizeof(struct mntent));
			}
		}
	}

	if (rootname)
	{
#if DEBUG && VERBOSE
		fprintf(stdout, "last: %s   %s   %s   %s   %d %d\n",
			fstabEntry.mnt_fsname,
			fstabEntry.mnt_dir,
			fstabEntry.mnt_type,
			fstabEntry.mnt_opts,
			fstabEntry.mnt_freq,
			fstabEntry.mnt_passno);
#endif
		addmntent(fstab, &fstabEntry);
	}
	
	endmntent(fstab);
	errno = errnosave;
	return ret;
}


Plugin *ELEKTRA_PLUGIN_EXPORT(fstab) {
	return elektraPluginExport("fstab",
		ELEKTRA_PLUGIN_GET,            &elektraFstabGet,
		ELEKTRA_PLUGIN_SET,            &elektraFstabSet,
		ELEKTRA_PLUGIN_END);
}


